# Mission Control v2 - Herontwerp Plan

## Gebruikerseisen

1. **Alleen open terminals tonen** - geen dev servers, alleen echte AI agents
2. **Volledige mapnaam** - ook subfolders zichtbaar
3. **Subtiele activiteit indicator** - knipperend bolletje als hartslag, niet hele kaart goud
4. **Claude, Codex EN Gemini** - alle AI terminals
5. **Runbook achtergrond processen** - headless agents zichtbaar
6. **Idle processen killen** - cleanup integratie
7. **Notificatie belletje per terminal** - aan/uit toggle, gaat af als agent klaar is
8. **Percentage meter verwijderen** - irrelevant

---

## Research Inzichten

### Dashboard Best Practices (2025)

- **Focus op wat de gebruiker in 5 seconden moet begrijpen** - niet overladen met data
- **F/Z scanning pattern** - belangrijkste info linksboven
- **Microinteractions** - subtiele animaties tonen dat systeem responsive is
- **Real-time resilience** - auto-retry met exponential backoff

### CLI Agent Status Detectie

- **IO tracking is onbetrouwbaar** - fluctueert constant
- **Process state detectie is moeilijk** - "running" vs "waiting for input" niet precies te onderscheiden
- **Transcript file monitoring** - betrouwbaarste methode voor activity detectie
- **Claude Code Statusline** - exposed JSON met sessie-info, maar geen directe working/idle state

---

## Feature Implementatie Plan

### 1. Strikte Agent Detectie

**Probleem:** Dev servers met "claude" in folder path worden gedetecteerd als terminals

**Oplossing:**
```python
AI_AGENTS = {
    "claude": {
        "patterns": ["claude-code/cli.js", "claude-code\\cli.js", "@anthropic-ai/claude"],
        "exclude": ["vite", "webpack", "next", "node_modules/.bin"]
    },
    "codex": {
        "patterns": ["codex-cli", "@openai/codex", "openai-codex"],
        "exclude": []
    },
    "gemini": {
        "patterns": ["gemini-cli", "@google/gemini", "gemini/cli"],
        "exclude": []
    }
}
```

**Bestanden:** `terminal_monitor.py`

---

### 2. Volledige Mapnaam

**Probleem:** Alleen laatste folder getoond

**Oplossing:**
```python
def format_path(cwd: str) -> str:
    home = os.path.expanduser("~")
    if cwd.startswith(home):
        return "~" + cwd[len(home):].replace("\\", "/")
    return cwd.replace("\\", "/")

# Voorbeeld:
# C:\Users\info\ai-stack\bmad_autopilot_kit_claude
# → ~/ai-stack/bmad_autopilot_kit_claude
```

**Bestanden:** `terminal_monitor.py`, `TerminalCard.tsx`

---

### 3. Activity Indicator (Hartslag)

**Probleem:** IO tracking fluctueert, hele kaart wordt goud

**Oplossing: Transcript File Monitoring**

```python
def get_activity_status(pid: int, transcript_path: str) -> bool:
    """Check of transcript file recent gewijzigd is."""
    try:
        mtime = os.path.getmtime(transcript_path)
        age = time.time() - mtime
        return age < 5.0  # Actief als < 5 seconden geleden
    except:
        return False
```

**UI Component:**
```tsx
// Subtiel knipperend bolletje
<span className={`
  w-2 h-2 rounded-full
  ${isActive
    ? 'bg-gold-400 animate-pulse shadow-[0_0_8px_rgba(250,204,21,0.6)]'
    : 'bg-slate-600'
  }
`} />
```

**Bestanden:** `terminal_monitor.py`, `TerminalCard.tsx`

---

### 4. Codex en Gemini Support

**Implementatie:**
- Voeg patterns toe aan detectie
- Verschillende kleuren per agent type:
  - Claude: gold
  - Codex: emerald
  - Gemini: blue

**Bestanden:** `terminal_monitor.py`, `TerminalCard.tsx`, `types.ts`

---

### 5. Notificatie Toggle per Terminal

**Data Model:**
```typescript
// localStorage key: 'terminal-notifications'
interface NotificationPrefs {
  [pid: number]: boolean;  // true = notify enabled
  defaultEnabled: boolean;
}
```

**UI:**
```tsx
<button
  onClick={() => toggleNotification(terminal.pid)}
  className="text-slate-400 hover:text-gold-300"
>
  {notifyEnabled ? <Bell /> : <BellOff />}
</button>
```

**Logic:**
```typescript
// Notify wanneer:
// 1. Was actief (isActive = true)
// 2. Nu niet meer actief (isActive = false)
// 3. Notification enabled voor deze terminal
// 4. Minstens 5 sec geen activiteit (debounce)
```

**Bestanden:** `useTerminalData.ts`, `TerminalCard.tsx`, nieuwe `useNotifications.ts`

---

### 6. CPU/Memory Verwijderen

**Actie:** Verwijder uit compact view, optioneel in expanded detail

**Bestanden:** `TerminalCard.tsx`

---

### 7. Background/Runbook Processen

**Detectie:**
```python
def is_background_process(proc) -> bool:
    """Check of process geen terminal window heeft."""
    try:
        # Op Windows: check of parent WindowsTerminal of conhost is
        parent = proc.parent()
        if parent:
            parent_name = parent.name().lower()
            if parent_name in ('windowsterminal.exe', 'conhost.exe', 'cmd.exe', 'powershell.exe'):
                return False
        return True  # Geen terminal parent = background
    except:
        return False
```

**UI:** Badge "BG" of "Runbook" naast terminal naam

**Bestanden:** `terminal_monitor.py`, `TerminalCard.tsx`

---

### 8. Kill Orphans Integratie

**UI:**
```tsx
{terminal.isOrphan && (
  <button
    onClick={() => killProcess(terminal.pid)}
    className="text-rose-400 hover:text-rose-300"
  >
    <X className="w-4 h-4" />
  </button>
)}
```

**API Endpoint:** `POST /api/terminal/<pid>/kill`

**Bestanden:** `dashboard_server.py`, `TerminalCard.tsx`

---

## Nieuw Data Model

```typescript
interface Terminal {
  pid: number;
  type: 'claude' | 'codex' | 'gemini';

  // Paths
  cwd: string;            // Volledige path
  cwdDisplay: string;     // ~/project/subfolder

  // Status
  isActive: boolean;      // Recent output < 5 sec
  isBackground: boolean;  // Geen terminal window
  isOrphan: boolean;      // Kan veilig gekillt worden

  // Timing
  startedAt: string;
  lastActivityAt: string;
  uptimeMinutes: number;
}

// Frontend-only state
interface TerminalUIState {
  notifyOnDone: boolean;
  isExpanded: boolean;
}
```

---

## Implementatie Volgorde

| # | Feature | Prioriteit | Effort |
|---|---------|------------|--------|
| 1 | Strikte agent detectie | HIGH | S |
| 2 | Volledige padnaam | HIGH | S |
| 3 | Activity indicator (transcript) | HIGH | M |
| 4 | Codex/Gemini support | MEDIUM | S |
| 5 | Notificatie toggle | HIGH | M |
| 6 | CPU/Memory verwijderen | LOW | XS |
| 7 | Background processen | MEDIUM | M |
| 8 | Kill integratie | LOW | S |

---

## Bestandswijzigingen

### Backend (`tools/story-runner/dashboard/`)
- `terminal_monitor.py` - Hoofdlogica herschrijven
- `dashboard_server.py` - Kill endpoint toevoegen

### Frontend (`tools/story-runner/dashboard-ui/src/`)
- `types.ts` - Nieuw data model
- `hooks/useTerminalData.ts` - Activity + notification logic
- `hooks/useNotifications.ts` - Nieuw: notification preferences
- `components/TerminalCard.tsx` - Nieuwe UI
- `components/ActivityDot.tsx` - Nieuw: hartslag component
- `App.tsx` - Layout cleanup

---

## Bronnen

- [Dashboard UX Best Practices 2025](https://www.smashingmagazine.com/2025/09/ux-strategies-real-time-dashboards/)
- [CLI UX Progress Displays](https://evilmartians.com/chronicles/cli-ux-best-practices-3-patterns-for-improving-progress-displays)
- [Claude Code Statusline Docs](https://code.claude.com/docs/en/statusline)
- [Process State Detection](https://unix.com/programming/76327-detecting-if-command-waiting-input.html)
