function qs(id) {
  return document.getElementById(id);
}

function badge(status) {
  const span = document.createElement("span");
  span.classList.add("badge");
  span.textContent = status;
  if (status === "running") span.classList.add("warn");
  else if (status === "finished") span.classList.add("good");
  else if (status === "failed") span.classList.add("bad");
  else if (status === "failed-but-merged") span.classList.add("warn");
  else span.classList.add("warn");
  return span;
}

async function jget(url) {
  const r = await fetch(url, { cache: "no-store" });
  if (!r.ok) throw new Error(`${r.status} ${r.statusText}`);
  return await r.json();
}

function setPills(container, items) {
  container.textContent = "";
  for (const it of items) {
    const pill = document.createElement("div");
    pill.className = "pill";
    pill.textContent = it;
    container.appendChild(pill);
  }
}

function fmtMinutesAgo(m) {
  if (m === null || m === undefined) return "—";
  if (m === 0) return "nu";
  return `${m}m`;
}

async function refreshAll() {
  const process = qs("process").value.trim();
  if (!process) return;

  qs("subtitle").textContent = `Proces: ${process}`;

  const [merged, plan, runs] = await Promise.all([
    jget(`/api/process/${process}/merged`),
    jget(`/api/process/${process}/plan`),
    jget(`/api/runs`),
  ]);

  const mergedOk = merged.merged_ok || [];
  const storyIds = plan.story_ids || [];
  const remaining = storyIds.filter((s) => !mergedOk.includes(s));

  qs("mergedCount").textContent = `${mergedOk.length}`;
  qs("totalCount").textContent = `${storyIds.length}`;
  qs("remainingCount").textContent = `${remaining.length}`;
  setPills(qs("mergedList"), mergedOk);

  const body = qs("runsBody");
  body.textContent = "";
  for (const r of runs.runs || []) {
    if (r.process !== process) continue;
    const tr = document.createElement("tr");

    const tdRun = document.createElement("td");
    tdRun.className = "mono";
    tdRun.textContent = r.run_id;

    const tdStatus = document.createElement("td");
    let label = r.overall_status || "unknown";
    const failedIds = r.failed_story_ids || [];
    if (label === "failed" && failedIds.some((s) => mergedOk.includes(s))) {
      label = "failed-but-merged";
    }
    tdStatus.appendChild(badge(label));

    const tdStory = document.createElement("td");
    tdStory.className = "mono";
    tdStory.textContent = r.current_story || "—";

    const tdPhase = document.createElement("td");
    tdPhase.className = "mono";
    tdPhase.textContent = r.phase || "—";

    const tdProg = document.createElement("td");
    tdProg.className = "mono";
    const idx = r.current_index || 0;
    const tot = r.total || 0;
    const doneCount = r.done_count || 0;
    tdProg.textContent = tot ? `${idx}/${tot} (done ${doneCount})` : "—";

    const tdHb = document.createElement("td");
    tdHb.textContent = fmtMinutesAgo(r.heartbeat_minutes_ago);

    const tdStoryMin = document.createElement("td");
    tdStoryMin.textContent = r.story_minutes_running === null ? "—" : `${r.story_minutes_running}m`;

    tr.append(tdRun, tdStatus, tdStory, tdPhase, tdProg, tdHb, tdStoryMin);
    body.appendChild(tr);
  }
}

function start() {
  const storedProcess = localStorage.getItem("sr_process") || "";
  const storedRefresh = localStorage.getItem("sr_refresh") || "5";
  qs("process").value = storedProcess;
  qs("refresh").value = storedRefresh;

  let timer = null;
  const schedule = () => {
    if (timer) clearInterval(timer);
    const seconds = Math.max(2, parseInt(qs("refresh").value || "5", 10));
    timer = setInterval(() => refreshAll().catch(() => {}), seconds * 1000);
  };

  qs("apply").addEventListener("click", () => {
    localStorage.setItem("sr_process", qs("process").value.trim());
    localStorage.setItem("sr_refresh", qs("refresh").value.trim());
    schedule();
    refreshAll().catch(() => {});
  });

  schedule();
  if (qs("process").value.trim()) refreshAll().catch(() => {});
}

start();
