from __future__ import annotations

import argparse
import json
import os
import re
import subprocess
import time
from dataclasses import dataclass
from datetime import datetime, timezone
from http.server import SimpleHTTPRequestHandler, ThreadingHTTPServer
from pathlib import Path
from urllib.parse import unquote


@dataclass(frozen=True)
class RunRef:
    process: str
    run_id: str
    status_path: Path


def _utc_now() -> float:
    return time.time()


def _read_json(path: Path) -> dict:
    try:
        return json.loads(path.read_text(encoding="utf-8"))
    except FileNotFoundError:
        return {}
    except json.JSONDecodeError:
        return {}


def _git(repo_root: Path, *args: str) -> str:
    p = subprocess.run(
        ["git", "-C", str(repo_root), *args],
        capture_output=True,
        text=True,
        check=False,
    )
    return (p.stdout or "").strip()


def _parse_story_ids_from_process_file(process_file: Path, stories_dir: Path) -> list[str]:
    if process_file.exists():
        txt = process_file.read_text(encoding="utf-8", errors="replace")
        if "## Canonical Story Order" in txt:
            ids: list[str] = []
            seen: set[str] = set()
            for line in txt.splitlines():
                for m in re.finditer(r"\b([A-Z]+-\d{3})\b", line):
                    sid = m.group(1)
                    if sid not in seen:
                        seen.add(sid)
                        ids.append(sid)
            if ids:
                return ids

    ids = []
    for p in stories_dir.glob("*.md"):
        sid = p.stem.strip()
        if re.fullmatch(r"[A-Z]+-\d{3}", sid):
            ids.append(sid)
    ids.sort(key=lambda s: int(s.split("-")[1]))
    return ids


def _list_runs(runs_root: Path) -> list[RunRef]:
    out: list[RunRef] = []
    if not runs_root.exists():
        return out
    for process_dir in runs_root.iterdir():
        if not process_dir.is_dir():
            continue
        for run_dir in process_dir.iterdir():
            if not run_dir.is_dir():
                continue
            status_path = run_dir / "status.json"
            if status_path.exists():
                out.append(RunRef(process=process_dir.name, run_id=run_dir.name, status_path=status_path))
    out.sort(key=lambda r: r.run_id, reverse=True)
    return out


def _merged_story_ids(repo_root: Path, integration_branch: str) -> list[str]:
    subjects = _git(repo_root, "log", integration_branch, "--format=%s", "--grep", "^merge: ")
    ids: set[str] = set()
    for line in subjects.splitlines():
        m = re.match(r"^merge:\s*(?P<id>[A-Z]+-\d{3})\s*$", line.strip())
        if m:
            ids.add(m.group("id"))
    return sorted(ids, key=lambda s: int(s.split("-")[1]))


def _minutes_since_iso(ts_iso: str) -> int | None:
    if not ts_iso:
        return None
    try:
        s = ts_iso.strip()
        if s.endswith("Z"):
            s = s[:-1] + "+00:00"
        dt = datetime.fromisoformat(s)
        if dt.tzinfo is None:
            dt = dt.replace(tzinfo=timezone.utc)
        now = datetime.now(timezone.utc)
        delta = now - dt.astimezone(timezone.utc)
        return max(0, int(delta.total_seconds() / 60))
    except Exception:
        return None


class Handler(SimpleHTTPRequestHandler):
    def __init__(self, *args, repo_root: Path, runs_root: Path, **kwargs):
        self._repo_root = repo_root
        self._runs_root = runs_root
        super().__init__(*args, directory=str(Path(__file__).parent), **kwargs)

    def end_headers(self) -> None:
        self.send_header("Cache-Control", "no-store, max-age=0")
        super().end_headers()

    def do_GET(self) -> None:  # noqa: N802
        path = unquote(self.path.split("?", 1)[0])
        if path == "/api/runs":
            return self._handle_runs()
        if path.startswith("/api/run/"):
            return self._handle_run(path)
        if path.startswith("/api/process/") and path.endswith("/merged"):
            return self._handle_process_merged(path)
        if path.startswith("/api/process/") and path.endswith("/plan"):
            return self._handle_process_plan(path)
        return super().do_GET()

    def _send_json(self, obj: object, status: int = 200) -> None:
        data = json.dumps(obj, indent=2).encode("utf-8")
        self.send_response(status)
        self.send_header("Content-Type", "application/json; charset=utf-8")
        self.send_header("Content-Length", str(len(data)))
        self.end_headers()
        self.wfile.write(data)

    def _handle_runs(self) -> None:
        now = _utc_now()
        runs = []
        for ref in _list_runs(self._runs_root):
            st = _read_json(ref.status_path)
            cur = st.get("current", {}) if isinstance(st, dict) else {}
            stories = st.get("stories", []) if isinstance(st, dict) else []
            done_ids: list[str] = []
            failed_ids: list[str] = []
            if isinstance(stories, list):
                for it in stories:
                    if not isinstance(it, dict):
                        continue
                    sid = str(it.get("id", "")).strip()
                    sst = str(it.get("status", "")).strip()
                    if sid and sst == "done":
                        done_ids.append(sid)
                    if sid and sst == "failed":
                        failed_ids.append(sid)
            last_hb = st.get("last_heartbeat_utc", "")
            last_up = st.get("last_update_utc", "")
            hb_mins = _minutes_since_iso(last_hb) if last_hb else None
            up_mins = _minutes_since_iso(last_up) if last_up else None

            story_mins = _minutes_since_iso(cur.get("story_started_utc", "")) if isinstance(cur, dict) else None
            phase_mins = _minutes_since_iso(cur.get("phase_started_utc", "")) if isinstance(cur, dict) else None

            runs.append(
                {
                    "process": ref.process,
                    "run_id": ref.run_id,
                    "overall_status": st.get("overall_status", "unknown"),
                    "current_story": cur.get("story_id", ""),
                    "current_index": cur.get("index", 0),
                    "total": cur.get("total", 0),
                    "phase": cur.get("phase", ""),
                    "heartbeat_minutes_ago": hb_mins,
                    "update_minutes_ago": up_mins,
                    "story_minutes_running": story_mins,
                    "phase_minutes_running": phase_mins,
                    "done_story_ids": done_ids,
                    "failed_story_ids": failed_ids,
                    "done_count": len(done_ids),
                    "failed_count": len(failed_ids),
                    "status_path": str(ref.status_path),
                }
            )
        return self._send_json({"now_epoch": now, "runs": runs})

    def _handle_run(self, path: str) -> None:
        parts = path.split("/")
        if len(parts) != 5:
            return self._send_json({"error": "bad_path"}, status=400)
        _, _, _, process, run_id = parts
        status_path = self._runs_root / process / run_id / "status.json"
        if not status_path.exists():
            return self._send_json({"error": "not_found"}, status=404)
        return self._send_json(_read_json(status_path))

    def _handle_process_merged(self, path: str) -> None:
        parts = path.split("/")
        if len(parts) != 5:
            return self._send_json({"error": "bad_path"}, status=400)
        process = parts[3]
        merged = _merged_story_ids(self._repo_root, process)
        return self._send_json({"process": process, "integration_branch": process, "merged_ok": merged})

    def _handle_process_plan(self, path: str) -> None:
        parts = path.split("/")
        if len(parts) != 5:
            return self._send_json({"error": "bad_path"}, status=400)
        process = parts[3]
        process_file = self._repo_root / "docs" / "processes" / process / "PROCESS.md"
        stories_dir = self._repo_root / "stories" / process
        story_ids = _parse_story_ids_from_process_file(process_file, stories_dir)
        return self._send_json({"process": process, "story_ids": story_ids, "total": len(story_ids)})


def main() -> int:
    ap = argparse.ArgumentParser()
    ap.add_argument("--repo-root", required=True)
    ap.add_argument("--port", type=int, default=4545)
    ns = ap.parse_args()

    repo_root = Path(ns.repo_root).resolve()
    runs_root = repo_root / "tools" / "story-runner" / "runs"
    host = "127.0.0.1"

    def handler(*args, **kwargs):
        return Handler(*args, repo_root=repo_root, runs_root=runs_root, **kwargs)

    httpd = ThreadingHTTPServer((host, ns.port), handler)
    print(f"Serving on http://{host}:{ns.port}/ (repo={repo_root})")
    httpd.serve_forever()
    return 0


if __name__ == "__main__":
    raise SystemExit(main())