"""
Cleanup Orphaned Claude Processes - SAFE VERSION

WHAT IT KILLS (safe to remove):
  - statusline.ps1 scripts (background helpers)
  - Shell snapshots (restore scripts)
  - Orphaned bash/pwsh shells spawned by Claude

WHAT IT NEVER KILLS (protected):
  - Real Claude Code CLI (node.exe with claude-code/cli.js)
  - Any process with an active terminal window
  - Vite/webpack dev servers (user started these intentionally)
"""
from __future__ import annotations

import os
import sys
import psutil

# Current process ID (don't kill ourselves!)
CURRENT_PID = os.getpid()
PARENT_PID = os.getppid()

# PROTECTED - Never kill these (real AI terminals)
PROTECTED_PATTERNS = [
    # Claude Code
    "claude-code/cli.js",
    "claude-code\\cli.js",
    "@anthropic-ai/claude",
    # Google Gemini CLI
    "gemini-cli",
    "@google/gemini",
    "gemini/cli",
    # OpenAI Codex CLI
    "codex-cli",
    "@openai/codex",
    "openai-codex",
]

# SAFE TO KILL - Background noise processes
ORPHAN_PATTERNS = [
    "statusline.ps1",      # Statusline helper scripts
    "shell-snapshots",     # Shell restore scripts
]


def is_protected(cmdline_str: str) -> bool:
    """Check if this is a PROTECTED process (real Claude terminal)."""
    cmdline_lower = cmdline_str.lower()
    for pattern in PROTECTED_PATTERNS:
        if pattern.lower() in cmdline_lower:
            return True
    return False


def is_orphan_candidate(proc: psutil.Process) -> tuple[bool, str]:
    """
    Check if process is an orphaned Claude-related process.
    Returns (is_orphan, reason) tuple.

    SAFE: Only returns True for background scripts, NEVER for real terminals.
    """
    try:
        name = proc.name().lower()
        cmdline = " ".join(proc.cmdline())
        pid = proc.pid

        # Never kill ourselves or parent
        if pid in (CURRENT_PID, PARENT_PID):
            return False, "current process"

        # NEVER kill protected processes (real Claude terminals)
        if is_protected(cmdline):
            return False, "PROTECTED: real Claude terminal"

        # Only kill known orphan patterns
        cmdline_lower = cmdline.lower()
        for pattern in ORPHAN_PATTERNS:
            if pattern.lower() in cmdline_lower:
                return True, f"orphan: {pattern}"

        return False, ""
    except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
        return False, ""


def find_orphans() -> list[tuple[int, str, str]]:
    """Find all orphaned Claude processes (SAFE - never finds real terminals)."""
    orphans = []

    for proc in psutil.process_iter(['pid', 'name']):
        try:
            is_orphan, reason = is_orphan_candidate(proc)
            if is_orphan:
                cmdline = " ".join(proc.cmdline())[:80]
                orphans.append((proc.pid, reason, cmdline))
        except (psutil.NoSuchProcess, psutil.AccessDenied):
            continue

    return orphans


def kill_orphans(dry_run: bool = False) -> int:
    """Kill orphaned processes. Returns count of killed processes."""
    orphans = find_orphans()

    if not orphans:
        print("[OK] No orphaned processes found!")
        print("     (Real Claude terminals are PROTECTED and never killed)")
        return 0

    print(f"[FOUND] {len(orphans)} orphaned background processes:")
    for pid, reason, cmdline in orphans:
        print(f"  PID {pid}: {reason}")
        print(f"      {cmdline}...")

    if dry_run:
        print("\n[DRY RUN] Would kill these background processes.")
        print("          Real Claude terminals are PROTECTED.")
        print("          Run with --kill to actually kill them.")
        return 0

    print("\n[KILLING] Terminating orphaned background processes...")
    killed = 0
    for pid, reason, _ in orphans:
        try:
            proc = psutil.Process(pid)
            proc.terminate()
            killed += 1
            print(f"  Killed PID {pid}")
        except (psutil.NoSuchProcess, psutil.AccessDenied) as e:
            print(f"  Failed to kill PID {pid}: {e}")

    print(f"\n[DONE] Killed {killed}/{len(orphans)} orphaned processes")
    print("       Real Claude terminals were NOT touched.")
    return killed


if __name__ == "__main__":
    dry_run = "--kill" not in sys.argv

    print("=" * 50)
    print("CLAUDE ORPHAN CLEANUP - SAFE VERSION")
    print("=" * 50)
    print()
    print("PROTECTED (never killed):")
    print("  - Real Claude Code terminals")
    print("  - Vite/webpack dev servers")
    print()
    print("WILL REMOVE (background noise):")
    print("  - statusline.ps1 scripts")
    print("  - shell-snapshots restore scripts")
    print()

    if dry_run:
        print("[DRY RUN MODE]\n")
    else:
        print("[KILL MODE]\n")

    kill_orphans(dry_run=dry_run)
