from __future__ import annotations

import argparse
import json
import os
import re
import subprocess
import time
from dataclasses import dataclass
from datetime import datetime, timezone
from http.server import SimpleHTTPRequestHandler, ThreadingHTTPServer
from pathlib import Path
from urllib.parse import unquote

# Terminal monitoring
from terminal_monitor import scan_terminals, get_terminal_by_pid, get_system_stats


@dataclass(frozen=True)
class RunRef:
    process: str
    run_id: str
    status_path: Path
    category: str  # 'process' or 'terminal'


def _utc_now() -> float:
    return time.time()


def _read_json(path: Path) -> dict:
    try:
        return json.loads(path.read_text(encoding="utf-8"))
    except FileNotFoundError:
        return {}
    except json.JSONDecodeError:
        return {}


def _git(repo_root: Path, *args: str) -> str:
    p = subprocess.run(
        ["git", "-C", str(repo_root), *args],
        capture_output=True,
        text=True,
        check=False,
    )
    return (p.stdout or "").strip()


def _parse_story_ids_from_process_file(process_file: Path, stories_dir: Path) -> list[str]:
    if process_file.exists():
        txt = process_file.read_text(encoding="utf-8", errors="replace")
        if "## Canonical Story Order" in txt:
            ids: list[str] = []
            seen: set[str] = set()
            for line in txt.splitlines():
                for m in re.finditer(r"\b([A-Z]+-\d{3})\b", line):
                    sid = m.group(1)
                    if sid not in seen:
                        seen.add(sid)
                        ids.append(sid)
            if ids:
                return ids

    ids = []
    for p in stories_dir.glob("*.md"):
        sid = p.stem.strip()
        if re.fullmatch(r"[A-Z]+-\d{3}", sid):
            ids.append(sid)
    ids.sort(key=lambda s: int(s.split("-")[1]))
    return ids


def _list_runs(runs_root: Path | list[Path]) -> list[RunRef]:
    out: list[RunRef] = []
    
    # Handle single path vs list of paths
    roots = [runs_root] if isinstance(runs_root, Path) else runs_root
    
    for root in roots:
        if not root.exists():
            continue
            
        # Scan for process directories (first level)
        for process_dir in root.iterdir():
            if not process_dir.is_dir():
                continue
                
            # Scan for run directories (second level)
            for run_dir in process_dir.iterdir():
                if not run_dir.is_dir():
                    continue
                    
                status_path = run_dir / "status.json"
                if status_path.exists():
                    data = _read_json(status_path)
                    category = data.get("type", "process")
                    
                    # Use the project name from the path if possible to disambiguate
                    project_name = "unknown"
                    try:
                        # Attempt to find the project root name relative to runs folder
                        # .../ProjectName/tools/story-runner/runs/...
                        project_name = root.parents[2].name
                    except IndexError:
                        pass
                        
                    display_process = process_dir.name
                    
                    out.append(RunRef(
                        process=display_process, 
                        run_id=run_dir.name, 
                        status_path=status_path,
                        category=category
                    ))
    
    # Sort by run_id (newest first)
    out.sort(key=lambda r: r.run_id, reverse=True)
    return out


def _merged_story_ids(repo_root: Path, integration_branch: str) -> list[str]:
    subjects = _git(repo_root, "log", integration_branch, "--format=%s", "--grep", "^merge: ")
    ids: set[str] = set()
    for line in subjects.splitlines():
        m = re.match(r"^merge:\s*(?P<id>[A-Z]+-\d{3})\s*$", line.strip())
        if m:
            ids.add(m.group("id"))
    return sorted(ids, key=lambda s: int(s.split("-")[1]))


def _minutes_since_iso(ts_iso: str) -> int | None:
    if not ts_iso:
        return None
    try:
        s = ts_iso.strip()
        if s.endswith("Z"):
            s = s[:-1] + "+00:00"
        dt = datetime.fromisoformat(s)
        if dt.tzinfo is None:
            dt = dt.replace(tzinfo=timezone.utc)
        now = datetime.now(timezone.utc)
        delta = now - dt.astimezone(timezone.utc)
        return max(0, int(delta.total_seconds() / 60))
    except Exception:
        return None


class Handler(SimpleHTTPRequestHandler):
    def __init__(self, *args, repo_root: Path, runs_root: Path, **kwargs):
        self._repo_root = repo_root
        self._runs_root = runs_root
        super().__init__(*args, directory=str(Path(__file__).parent), **kwargs)

    def end_headers(self) -> None:
        self.send_header("Cache-Control", "no-store, max-age=0")
        super().end_headers()

    def do_GET(self) -> None:  # noqa: N802
        path = unquote(self.path.split("?", 1)[0])
        # Terminal monitor endpoints (new)
        if path == "/api/terminals":
            return self._handle_terminals()
        if path.startswith("/api/terminal/"):
            return self._handle_terminal_detail(path)
        if path == "/api/system":
            return self._handle_system_stats()
        # Legacy story runner endpoints
        if path == "/api/runs":
            return self._handle_runs()
        if path.startswith("/api/run/"):
            return self._handle_run(path)
        if path.startswith("/api/process/") and path.endswith("/merged"):
            return self._handle_process_merged(path)
        if path.startswith("/api/process/") and path.endswith("/plan"):
            return self._handle_process_plan(path)
        return super().do_GET()

    def _send_json(self, obj: object, status: int = 200) -> None:
        data = json.dumps(obj, indent=2).encode("utf-8")
        self.send_response(status)
        self.send_header("Content-Type", "application/json; charset=utf-8")
        self.send_header("Content-Length", str(len(data)))
        self.end_headers()
        self.wfile.write(data)

    def _handle_runs(self) -> None:
        now = _utc_now()
        runs = []
        for ref in _list_runs(self._runs_root):
            st = _read_json(ref.status_path)
            cur = st.get("current", {}) if isinstance(st, dict) else {}
            stories = st.get("stories", []) if isinstance(st, dict) else []
            done_ids: list[str] = []
            failed_ids: list[str] = []
            if isinstance(stories, list):
                for it in stories:
                    if not isinstance(it, dict):
                        continue
                    sid = str(it.get("id", "")).strip()
                    sst = str(it.get("status", "")).strip()
                    if sid and sst == "done":
                        done_ids.append(sid)
                    if sid and sst == "failed":
                        failed_ids.append(sid)
            
            last_hb = st.get("last_heartbeat_utc", "")
            last_up = st.get("last_update_utc", "")
            hb_mins = _minutes_since_iso(last_hb) if last_hb else None
            up_mins = _minutes_since_iso(last_up) if last_up else None

            story_mins = _minutes_since_iso(cur.get("story_started_utc", "")) if isinstance(cur, dict) else None
            phase_mins = _minutes_since_iso(cur.get("phase_started_utc", "")) if isinstance(cur, dict) else None

            runs.append(
                {
                    "process": ref.process,
                    "run_id": ref.run_id,
                    "category": ref.category,
                    "command": st.get("command", ""),
                    "agent": st.get("agent", "user"),
                    "overall_status": st.get("overall_status", "unknown"),
                    "current_story": cur.get("story_id", ""),
                    "current_index": cur.get("index", 0),
                    "total": cur.get("total", 0),
                    "phase": cur.get("phase", ""),
                    "heartbeat_minutes_ago": hb_mins,
                    "update_minutes_ago": up_mins,
                    "story_minutes_running": story_mins,
                    "phase_minutes_running": phase_mins,
                    "done_story_ids": done_ids,
                    "failed_story_ids": failed_ids,
                    "done_count": len(done_ids),
                    "failed_count": len(failed_ids),
                    "duration_seconds": st.get("duration_seconds"),
                    "status_path": str(ref.status_path),
                }
            )
        return self._send_json({"now_epoch": now, "runs": runs})

    def _handle_run(self, path: str) -> None:
        parts = path.split("/")
        if len(parts) != 5:
            return self._send_json({"error": "bad_path"}, status=400)
        _, _, _, process, run_id = parts
        status_path = self._runs_root / process / run_id / "status.json"
        if not status_path.exists():
            return self._send_json({"error": "not_found"}, status=404)
        return self._send_json(_read_json(status_path))

    def _handle_process_merged(self, path: str) -> None:
        parts = path.split("/")
        if len(parts) != 5:
            return self._send_json({"error": "bad_path"}, status=400)
        process = parts[3]
        merged = _merged_story_ids(self._repo_root, process)
        return self._send_json({"process": process, "integration_branch": process, "merged_ok": merged})

    def _handle_process_plan(self, path: str) -> None:
        parts = path.split("/")
        if len(parts) != 5:
            return self._send_json({"error": "bad_path"}, status=400)
        process = parts[3]
        process_file = self._repo_root / "docs" / "processes" / process / "PROCESS.md"
        stories_dir = self._repo_root / "stories" / process
        story_ids = _parse_story_ids_from_process_file(process_file, stories_dir)
        return self._send_json({"process": process, "story_ids": story_ids, "total": len(story_ids)})

    # ============ Terminal Monitor Endpoints ============

    def _handle_terminals(self) -> None:
        """GET /api/terminals - List all active terminals."""
        now = _utc_now()
        terminals = scan_terminals()
        return self._send_json({
            "now_epoch": now,
            "terminals": [t.to_dict() for t in terminals],
            "count": len(terminals),
        })

    def _handle_terminal_detail(self, path: str) -> None:
        """GET /api/terminal/<pid> - Get details for specific terminal."""
        parts = path.split("/")
        if len(parts) != 4:
            return self._send_json({"error": "bad_path"}, status=400)
        try:
            pid = int(parts[3])
        except ValueError:
            return self._send_json({"error": "invalid_pid"}, status=400)

        terminal = get_terminal_by_pid(pid)
        if terminal is None:
            return self._send_json({"error": "not_found"}, status=404)
        return self._send_json(terminal.to_dict())

    def _handle_system_stats(self) -> None:
        """GET /api/system - Get system resource stats."""
        stats = get_system_stats()
        stats["now_epoch"] = _utc_now()
        return self._send_json(stats)


def main() -> int:
    ap = argparse.ArgumentParser()
    ap.add_argument("--repo-root", required=False, help="Legacy argument, ignores auto-discovery if set")
    ap.add_argument("--port", type=int, default=4545)
    ap.add_argument("--scan-parent", action="store_true", default=True, help="Scan sibling directories for other projects")
    ns = ap.parse_args()

    host = "127.0.0.1"
    scan_roots: list[Path] = []

    # 1. Determine where we are
    current_dir = Path.cwd()
    
    # 2. Global Discovery Mode (Sibling Scanner)
    if ns.scan_parent:
        print(f"[SCAN] Mission Control: Scanning for Story Runner projects in {current_dir.parent}...")
        # Look for sibling folders that have the story-runner structure
        # We assume structure: <UserDir>/<Project>/tools/story-runner/runs

        # If running from inside mission-control-v2 (current_dir is repo root)
        # Sibling search starts at current_dir.parent (C:\Users\info)
        search_start = current_dir.parent

        if search_start.exists():
            for project_dir in search_start.iterdir():
                if project_dir.is_dir():
                    potential_runs = project_dir / "tools" / "story-runner" / "runs"
                    if potential_runs.exists():
                        print(f"   -> Found signal in: {project_dir.name}")
                        scan_roots.append(potential_runs)

    # 3. Fallback / Local override
    if ns.repo_root:
        manual_root = Path(ns.repo_root).resolve() / "tools" / "story-runner" / "runs"
        if manual_root.exists():
             scan_roots.append(manual_root)

    if not scan_roots:
        print("[WARN] No active signal sources found. Waiting for missions to start...")

    def handler(*args, **kwargs):
        # We pass the LIST of roots to the handler
        return Handler(*args, repo_root=current_dir, runs_root=scan_roots, **kwargs)

    httpd = ThreadingHTTPServer((host, ns.port), handler)
    print(f"[OK] Mission Control Online on http://{host}:{ns.port}/")
    print(f"[INFO] Monitoring {len(scan_roots)} sectors.")
    httpd.serve_forever()
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
