"""
Terminal Monitor - Universal Process Scanner for Mission Control
Uses psutil to detect and monitor terminal processes on Windows.
"""
from __future__ import annotations

import time
import threading
from dataclasses import dataclass, asdict
from datetime import datetime, timezone
from typing import Optional

import psutil


# Simple cache for terminals with TTL
_cache_lock = threading.Lock()
_cached_terminals: list = []
_cache_time: float = 0
_CACHE_TTL: float = 0.5  # Cache valid for 0.5 seconds (faster updates for activity)

# IO tracking for activity detection
# pid -> (last_write_bytes, last_check_time, last_working_time)
_io_tracker: dict[int, tuple[int, float, float]] = {}
_IO_ACTIVITY_THRESHOLD: int = 5000  # Bytes written per second to be "working"
_WORKING_COOLDOWN: float = 3.0  # Stay "working" for at least 3 seconds after activity


# Fast lookup for terminal process names (lowercase)
TERMINAL_NAMES = frozenset([
    "powershell.exe", "pwsh.exe", "cmd.exe",
    "windowsterminal.exe", "conhost.exe",
    "node.exe", "bash.exe", "sh.exe", "zsh.exe",
    "code.exe",
])

TERMINAL_EXECUTABLES = {
    "powershell.exe": "powershell",
    "pwsh.exe": "powershell",
    "cmd.exe": "cmd",
    "windowsterminal.exe": "windows_terminal",
    "conhost.exe": "conhost",
    "bash.exe": "powershell",  # Git Bash treated as shell
    "sh.exe": "powershell",
    "zsh.exe": "powershell",
}

# AI Terminal CLI patterns - maps pattern to terminal type
AI_CLI_PATTERNS = {
    # Claude Code
    "claude-code/cli.js": "claude",
    "claude-code\\cli.js": "claude",
    "@anthropic-ai/claude": "claude",
    # Google Gemini CLI
    "gemini-cli": "gemini",
    "@google/gemini": "gemini",
    "gemini/cli": "gemini",
    # OpenAI Codex CLI
    "codex-cli": "codex",
    "@openai/codex": "codex",
    "openai-codex": "codex",
}

# NOT Claude - dev servers in folders with "claude" in name
DEV_SERVER_PATTERNS = [
    "vite.js",
    "vite/bin",
    "node_modules/.bin",
    "node_modules\\.bin",
    "webpack",
    "next dev",
    "react-scripts",
]

# Patterns that indicate NOISE (not real terminals user cares about)
NOISE_PATTERNS = [
    "statusline.ps1",
    "shell-snapshots",
    "\\usr\\bin\\bash",
    "/usr/bin/bash",
    "-c -l source",
]


@dataclass
class TerminalInfo:
    pid: int
    ppid: int
    type: str  # powershell | cmd | claude | vscode | windows_terminal | other
    name: str
    status: str  # active | idle | busy | working
    cwd: str
    cmdline: str
    cpu_percent: float
    memory_mb: float
    started_at: str
    uptime_minutes: int
    parent_terminal: Optional[str]
    children: list[dict]
    active_children: int  # Number of active child processes (indicator of work)
    io_read_bytes: int  # IO read bytes for activity tracking
    io_write_bytes: int  # IO write bytes for activity tracking

    def to_dict(self) -> dict:
        return asdict(self)


def _get_process_info(proc: psutil.Process, skip_cwd: bool = False) -> Optional[dict]:
    """Extract safe process info, handling access denied gracefully."""
    try:
        with proc.oneshot():
            # Get basic info first (fast)
            pid = proc.pid
            ppid = proc.ppid()
            name = proc.name()

            # Get cmdline (can fail)
            try:
                cmdline = proc.cmdline()
            except (psutil.AccessDenied, psutil.NoSuchProcess):
                cmdline = []

            # Get cwd only if needed (this is the SLOW operation)
            cwd = None
            if not skip_cwd:
                try:
                    cwd = proc.cwd()
                except (psutil.AccessDenied, psutil.NoSuchProcess, OSError):
                    pass

            # Get memory info
            try:
                memory_info = proc.memory_info()
            except (psutil.AccessDenied, psutil.NoSuchProcess):
                return None

            # Get create time
            try:
                create_time = proc.create_time()
            except (psutil.AccessDenied, psutil.NoSuchProcess):
                create_time = time.time()

            return {
                "pid": pid,
                "ppid": ppid,
                "name": name,
                "cmdline": cmdline,
                "cwd": cwd,
                "cpu_percent": 0.0,  # Skip CPU for speed
                "memory_info": memory_info,
                "create_time": create_time,
                "status": "running",
            }
    except (psutil.AccessDenied, psutil.NoSuchProcess, psutil.ZombieProcess, OSError):
        return None


def _is_noise(cmdline_str: str) -> bool:
    """Check if this is a noise process we should skip."""
    for pattern in NOISE_PATTERNS:
        if pattern.lower() in cmdline_str.lower():
            return True
    return False


def _detect_terminal_type(name: str, cmdline: list[str]) -> str:
    """Determine the terminal type from process name and command line."""
    name_lower = name.lower()
    cmdline_str = " ".join(cmdline).lower()

    # Skip noise processes (statuslines, shell snapshots, etc.)
    if _is_noise(cmdline_str):
        return "noise"

    # AI Terminal detection (Claude, Gemini, Codex)
    if name_lower in ("node.exe", "node"):
        # First check if it's a dev server (vite, webpack, etc) - skip
        if any(pattern.lower() in cmdline_str for pattern in DEV_SERVER_PATTERNS):
            return "other"
        # Check for AI CLI patterns
        for pattern, terminal_type in AI_CLI_PATTERNS.items():
            if pattern.lower() in cmdline_str:
                return terminal_type
        return "other"  # Other node processes, skip

    # VS Code terminal detection
    if "code" in name_lower or "code.exe" in name_lower:
        return "vscode"

    # Skip shells - they're just subprocesses, not real terminals
    if name_lower in ("bash.exe", "sh.exe", "zsh.exe"):
        return "noise"

    # Standard terminal detection (PowerShell, CMD)
    if name_lower in TERMINAL_EXECUTABLES:
        return TERMINAL_EXECUTABLES[name_lower]

    return "other"


def _get_parent_terminal_name(ppid: int) -> Optional[str]:
    """Get the name of the parent terminal if it's a known terminal type."""
    try:
        parent = psutil.Process(ppid)
        parent_name = parent.name().lower()
        if parent_name in TERMINAL_EXECUTABLES:
            return TERMINAL_EXECUTABLES[parent_name]
        if parent_name == "windowsterminal.exe":
            return "Windows Terminal"
        return None
    except (psutil.AccessDenied, psutil.NoSuchProcess):
        return None


def _get_child_processes(pid: int) -> list[dict]:
    """Get list of child processes for a given PID."""
    children = []
    try:
        parent = psutil.Process(pid)
        for child in parent.children(recursive=False):
            try:
                with child.oneshot():
                    children.append({
                        "pid": child.pid,
                        "name": child.name(),
                        "cmdline": " ".join(child.cmdline())[:100],
                        "cpu_percent": child.cpu_percent(),
                        "memory_mb": round(child.memory_info().rss / 1024 / 1024, 1),
                    })
            except (psutil.AccessDenied, psutil.NoSuchProcess):
                continue
    except (psutil.AccessDenied, psutil.NoSuchProcess):
        pass
    return children


def _get_activity_info(pid: int) -> tuple[bool, int, int, list[dict]]:
    """
    Get activity indicators for a process.
    Returns (is_working, io_read_bytes, io_write_bytes, children_list)

    Activity detection for Claude:
    - Track IO write bytes over time
    - If write bytes increased significantly since last check -> working
    - This catches actual file writes, API calls, etc.
    """
    global _io_tracker

    is_working = False
    io_read = 0
    io_write = 0
    children = []
    now = time.time()

    try:
        proc = psutil.Process(pid)

        # Get IO counters (indicates read/write activity)
        try:
            io = proc.io_counters()
            io_read = io.read_bytes
            io_write = io.write_bytes

            # Check if IO increased since last poll
            if pid in _io_tracker:
                last_write, last_time, last_working = _io_tracker[pid]
                time_delta = now - last_time
                if time_delta > 0.1:  # Minimum 100ms between checks
                    write_delta = io_write - last_write
                    # Calculate bytes per second
                    write_rate = write_delta / time_delta if time_delta > 0 else 0

                    # Working if write rate exceeds threshold
                    if write_rate > _IO_ACTIVITY_THRESHOLD:
                        is_working = True
                        last_working = now  # Update last working time
                    # Stay "working" during cooldown period
                    elif (now - last_working) < _WORKING_COOLDOWN:
                        is_working = True

                    # Update tracker with new last_working time
                    _io_tracker[pid] = (io_write, now, last_working)
                else:
                    # Too soon, keep previous state
                    if (now - last_working) < _WORKING_COOLDOWN:
                        is_working = True
            else:
                # First poll: initialize with current time as last_working (will expire)
                _io_tracker[pid] = (io_write, now, 0.0)

        except (psutil.AccessDenied, psutil.NoSuchProcess):
            pass

        # Build child info for direct children only
        for child in proc.children(recursive=False):
            try:
                with child.oneshot():
                    child_name = child.name()
                    children.append({
                        "pid": child.pid,
                        "name": child_name,
                        "cmdline": " ".join(child.cmdline())[:100],
                        "cpu_percent": 0.0,
                        "memory_mb": round(child.memory_info().rss / 1024 / 1024, 1),
                    })
            except (psutil.AccessDenied, psutil.NoSuchProcess):
                continue

    except (psutil.AccessDenied, psutil.NoSuchProcess):
        pass

    return is_working, io_read, io_write, children


def _format_uptime(create_time: float) -> int:
    """Calculate uptime in minutes from process creation time."""
    now = time.time()
    return int((now - create_time) / 60)


def _format_timestamp(create_time: float) -> str:
    """Format process creation time as ISO timestamp."""
    dt = datetime.fromtimestamp(create_time, tz=timezone.utc)
    return dt.isoformat()


def _determine_status(cpu_percent: float) -> str:
    """Determine terminal status based on CPU usage."""
    if cpu_percent > 10:
        return "busy"
    elif cpu_percent > 0.5:
        return "active"
    return "idle"


def scan_terminals(max_shells: int = 20) -> list[TerminalInfo]:
    """
    Scan for all terminal processes and return their info.
    Optimized for speed - uses caching and early filtering.

    Args:
        max_shells: Maximum number of non-Claude shells to return (default 20)
    """
    global _cached_terminals, _cache_time

    # Check cache first - but for AI terminals we need fresh status
    now = time.time()
    with _cache_lock:
        cache_age = now - _cache_time
        if _cached_terminals and cache_age < _CACHE_TTL:
            # Update AI terminal status (IO tracking needs fresh data)
            updated = []
            for t in _cached_terminals:
                if t.type in ("claude", "gemini", "codex"):
                    is_working, io_read, io_write, _ = _get_activity_info(t.pid)
                    new_status = "working" if is_working else ("active" if t.children else "idle")
                    updated.append(TerminalInfo(
                        pid=t.pid, ppid=t.ppid, type=t.type, name=t.name,
                        status=new_status, cwd=t.cwd, cmdline=t.cmdline,
                        cpu_percent=t.cpu_percent, memory_mb=t.memory_mb,
                        started_at=t.started_at, uptime_minutes=t.uptime_minutes,
                        parent_terminal=t.parent_terminal, children=t.children,
                        active_children=t.active_children,
                        io_read_bytes=io_read, io_write_bytes=io_write,
                    ))
                else:
                    updated.append(t)
            return updated

    claude_terminals: list[TerminalInfo] = []
    shell_terminals: list[TerminalInfo] = []
    seen_pids: set[int] = set()

    # Fast pass: only get name to filter quickly
    for proc in psutil.process_iter(['pid', 'name']):
        try:
            info = proc.info
            name = info.get('name')
            if not name:
                continue
            name_lower = name.lower()

            # Early exit: skip if not a potential terminal
            if name_lower not in TERMINAL_NAMES:
                continue

            pid = info.get('pid')
            if pid in seen_pids:
                continue
            seen_pids.add(pid)

            # Now get cmdline only for matching processes
            try:
                cmdline = proc.cmdline()
            except (psutil.AccessDenied, psutil.NoSuchProcess):
                cmdline = []

            terminal_type = _detect_terminal_type(name, cmdline)

            # Skip noise, Windows Terminal, conhost, and other non-terminals
            if terminal_type in ("windows_terminal", "conhost", "other", "noise"):
                continue

            is_claude = terminal_type == "claude"

            # Skip shell terminals if we already have enough
            if not is_claude and len(shell_terminals) >= max_shells:
                continue

            # Check if this is an AI terminal (Claude, Gemini, Codex)
            is_ai_terminal = terminal_type in ("claude", "gemini", "codex")

            # Get full info - only get cwd for AI terminals (it's slow)
            full_info = _get_process_info(proc, skip_cwd=not is_ai_terminal)
            if full_info is None:
                continue

            # Build terminal info
            cwd = full_info["cwd"] or "Unknown"
            cmdline_str = " ".join(cmdline) if cmdline else name

            # Truncate long command lines
            if len(cmdline_str) > 200:
                cmdline_str = cmdline_str[:197] + "..."

            memory_mb = round(full_info["memory_info"].rss / 1024 / 1024, 1)

            # Get activity info for AI terminals (IO delta tracking)
            is_working = False
            io_read = 0
            io_write = 0
            children = []
            status = "idle"

            if is_ai_terminal:
                is_working, io_read, io_write, children = _get_activity_info(pid)
                # Determine status based on IO activity
                if is_working:
                    status = "working"
                elif len(children) > 0:
                    status = "active"

            terminal = TerminalInfo(
                pid=pid,
                ppid=full_info["ppid"],
                type=terminal_type,
                name=name,
                status=status,
                cwd=cwd,
                cmdline=cmdline_str,
                cpu_percent=0.0,
                memory_mb=memory_mb,
                started_at=_format_timestamp(full_info["create_time"]),
                uptime_minutes=_format_uptime(full_info["create_time"]),
                parent_terminal=None,  # Skip parent lookup for speed
                children=children,
                active_children=len(children),
                io_read_bytes=io_read,
                io_write_bytes=io_write,
            )

            if is_ai_terminal:
                claude_terminals.append(terminal)
            else:
                shell_terminals.append(terminal)

        except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
            continue

    # Cache and return
    result = claude_terminals + shell_terminals
    with _cache_lock:
        _cached_terminals = result
        _cache_time = time.time()

    return result


def get_terminal_by_pid(pid: int) -> Optional[TerminalInfo]:
    """Get detailed info for a specific terminal by PID."""
    try:
        proc = psutil.Process(pid)
        info = _get_process_info(proc)
        if info is None:
            return None

        name = info["name"]
        cmdline = info["cmdline"] or []
        terminal_type = _detect_terminal_type(name, cmdline)

        cwd = info["cwd"] or "Unknown"
        cmdline_str = " ".join(cmdline) if cmdline else name

        memory_mb = round(info["memory_info"].rss / 1024 / 1024, 1)
        cpu = info["cpu_percent"]

        return TerminalInfo(
            pid=pid,
            ppid=info["ppid"],
            type=terminal_type,
            name=name,
            status=_determine_status(cpu),
            cwd=cwd,
            cmdline=cmdline_str,
            cpu_percent=round(cpu, 1),
            memory_mb=memory_mb,
            started_at=_format_timestamp(info["create_time"]),
            uptime_minutes=_format_uptime(info["create_time"]),
            parent_terminal=_get_parent_terminal_name(info["ppid"]),
            children=_get_child_processes(pid),
        )
    except (psutil.AccessDenied, psutil.NoSuchProcess):
        return None


def get_system_stats() -> dict:
    """Get overall system resource stats."""
    cpu = psutil.cpu_percent(interval=0.1)
    mem = psutil.virtual_memory()
    return {
        "cpu_percent": round(cpu, 1),
        "memory_total_gb": round(mem.total / 1024 / 1024 / 1024, 1),
        "memory_used_gb": round(mem.used / 1024 / 1024 / 1024, 1),
        "memory_percent": round(mem.percent, 1),
    }


if __name__ == "__main__":
    # Quick test
    import json
    print("Scanning terminals...")
    terminals = scan_terminals()
    print(f"Found {len(terminals)} terminals:\n")
    for t in terminals:
        print(json.dumps(t.to_dict(), indent=2))
    print("\nSystem stats:")
    print(json.dumps(get_system_stats(), indent=2))
