import React from 'react';
import { TerminalSidebar } from './components/TerminalSidebar';
import { TerminalCard } from './components/TerminalCard';
import { AudioManager } from './components/AudioManager';
import { Loader2, Sparkles, Terminal, Zap } from 'lucide-react';
import { useTerminalData } from './hooks/useTerminalData';
import logo from './assets/logo.png';

const App: React.FC = () => {
  const {
    terminals,
    claudeTerminals,
    shellTerminals,
    otherTerminals,
    systemStats,
    selectedPid,
    setSelectedPid,
    selectedTerminal,
    isLoading,
    error,
    stats,
    lastDoneEvent,
    workingCount,
  } = useTerminalData();

  // No auto-select - user clicks to expand a terminal

  return (
    <div className="flex h-screen bg-galaxy-950 text-slate-100 font-body overflow-hidden relative selection:bg-gold-300/30">

      {/* GALAXY BACKGROUND */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none flex items-center justify-center">
        {/* Vortex Layer */}
        {[...Array(30)].map((_, i) => (
          <div
            key={`vortex-${i}`}
            className="star-layer animate-vortex"
            style={{
              width: Math.random() > 0.8 ? '1.5px' : '0.8px',
              height: Math.random() > 0.8 ? '1.5px' : '0.8px',
              animationDelay: `${Math.random() * 6}s`,
              animationDuration: `${4 + Math.random() * 4}s`,
              left: `${50 + (Math.random() * 100 - 50)}%`,
              top: `${50 + (Math.random() * 100 - 50)}%`,
            }}
          />
        ))}
        {/* Twinkling Static Layer */}
        {[...Array(40)].map((_, i) => (
          <div
            key={`twinkle-${i}`}
            className="absolute bg-gold-200 rounded-full animate-twinkling opacity-20"
            style={{
              width: Math.random() * 2 + 'px',
              height: Math.random() * 2 + 'px',
              top: Math.random() * 100 + '%',
              left: Math.random() * 100 + '%',
              animationDelay: Math.random() * 5 + 's',
              boxShadow: '0 0 4px rgba(253, 230, 138, 0.5)'
            }}
          />
        ))}
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_center,transparent_20%,#02040a_100%)]" />
      </div>

      <TerminalSidebar
        terminals={terminals}
        selectedPid={selectedPid}
        onSelect={setSelectedPid}
        systemStats={systemStats}
        stats={stats}
      />

      <main className="flex-1 flex flex-col relative z-20 overflow-y-auto scrollbar-none">
        <header className="p-8 flex items-center justify-between border-b border-gold-300/5 backdrop-blur-md sticky top-0 z-20 bg-galaxy-950/40">
          <div>
            <h2 className="text-3xl font-heading font-bold tracking-tight mb-1 flex items-center gap-2 text-gold-gradient">
              Terminal Monitor
              {claudeTerminals.length > 0 && (
                <span className="w-2 h-2 bg-gold-400 rounded-full animate-ping shadow-[0_0_10px_#FACC15]" />
              )}
            </h2>
            <p className="text-slate-400 text-sm font-light">
              <span className="text-gold-300 font-medium">{stats.claudeCount}</span> AI instances
              {workingCount > 0 && (
                <>
                  {' '}&bull;{' '}
                  <span className="inline-flex items-center gap-1 text-gold-400">
                    <Zap className="w-3 h-3 animate-pulse" />
                    <span className="font-medium">{workingCount} working</span>
                  </span>
                </>
              )}
              {' '}&bull;{' '}
              <span className="text-slate-300">{stats.totalCount}</span> total terminals
            </p>
          </div>

          <div className="flex items-center">
             <div className="h-[72px] w-auto opacity-90 drop-shadow-[0_0_10px_rgba(253,230,138,0.2)]">
                <img src={logo} alt="JVDP" className="h-full w-auto object-contain" />
             </div>
          </div>
        </header>

        <div className="p-8 max-w-6xl w-full mx-auto space-y-8">
          {isLoading ? (
            <div className="h-64 flex flex-col items-center justify-center">
              <Loader2 className="w-8 h-8 text-gold-300 animate-spin mb-4" />
              <p className="text-slate-500">Scanning terminals...</p>
            </div>
          ) : error ? (
            <div className="h-64 flex flex-col items-center justify-center border-2 border-dashed border-rose-800 rounded-3xl bg-rose-950/20">
              <p className="text-rose-400 font-medium">Error: {error}</p>
              <p className="text-slate-500 text-sm mt-2">Make sure the backend is running on port 4545</p>
            </div>
          ) : terminals.length === 0 ? (
            <div className="h-64 flex flex-col items-center justify-center border-2 border-dashed border-galaxy-800 rounded-3xl">
              <Terminal className="w-8 h-8 text-slate-600 mb-4" />
              <p className="text-slate-500 font-medium">No terminals detected</p>
            </div>
          ) : (
            <>
              {/* HERO SECTION - Selected Terminal (click to collapse) */}
              {selectedTerminal && (
                <div className="space-y-4">
                  <label className="text-[10px] uppercase tracking-[0.2em] text-gold-300 font-black block pl-1 opacity-80">
                    {selectedTerminal?.type === 'claude' ? 'CLAUDE INSTANCE' : 'SELECTED TERMINAL'}
                    <span className="text-slate-500 ml-2">(click to collapse)</span>
                  </label>
                  <TerminalCard
                    terminal={selectedTerminal}
                    isSelected
                    isHero
                    onClick={() => setSelectedPid(null)}
                  />
                </div>
              )}

              {/* CLAUDE INSTANCES */}
              {claudeTerminals.length > 0 && (
                <div className="space-y-4">
                  <div className="flex items-center gap-2">
                    <Sparkles className="w-4 h-4 text-gold-300" />
                    <label className="text-[10px] uppercase tracking-[0.2em] text-gold-300/80 font-black">
                      Claude Instances ({claudeTerminals.length})
                    </label>
                  </div>
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-3">
                    {claudeTerminals.map(t => (
                      <TerminalCard
                        key={t.pid}
                        terminal={t}
                        isSelected={t.pid === selectedPid}
                        onClick={() => setSelectedPid(t.pid === selectedPid ? null : t.pid)}
                      />
                    ))}
                  </div>
                </div>
              )}

              {/* SHELL TERMINALS */}
              {shellTerminals.length > 0 && (
                <div className="space-y-4 pt-8 border-t border-gold-300/5">
                  <div className="flex items-center gap-2">
                    <Terminal className="w-4 h-4 text-slate-500" />
                    <label className="text-[10px] uppercase tracking-[0.2em] text-slate-400 font-black">
                      Shell Terminals ({shellTerminals.length})
                    </label>
                  </div>
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-3">
                    {shellTerminals.slice(0, 6).map(t => (
                      <TerminalCard
                        key={t.pid}
                        terminal={t}
                        isSelected={t.pid === selectedPid}
                        onClick={() => setSelectedPid(t.pid === selectedPid ? null : t.pid)}
                      />
                    ))}
                  </div>
                  {shellTerminals.length > 6 && (
                    <p className="text-xs text-slate-600 text-center">
                      +{shellTerminals.length - 6} more terminals
                    </p>
                  )}
                </div>
              )}
            </>
          )}
        </div>
      </main>

      <AudioManager lastEvent={lastDoneEvent ? 'done' : undefined} />
    </div>
  );
};

export default App;
