import React from 'react';
import { X, Copy, Terminal } from 'lucide-react';

interface ErrorDrawerProps {
  isOpen: boolean;
  onClose: () => void;
  failure?: {
    runId: string;
    storyId: string;
    reason: string;
    logs: string;
  };
}

export const ErrorDrawer: React.FC<ErrorDrawerProps> = ({ isOpen, onClose, failure }) => {
  if (!failure) return null;

  const copyForLLM = () => {
    const text = `
    STORY RUNNER FAILURE REPORT
    Run ID: ${failure.runId}
    Story ID: ${failure.storyId}
    Reason: ${failure.reason}
    
    Logs:
    ${failure.logs}
    `.trim();
    navigator.clipboard.writeText(text);
    alert('Copied to clipboard for LLM context!');
  };

  return (
    <div 
      className={`fixed top-0 right-0 h-full w-full max-w-md bg-slate-950/90 border-l border-rose-500/30 backdrop-blur-xl transition-transform duration-500 z-50 ${
        isOpen ? 'translate-x-0' : 'translate-x-full'
      }`}
    >
      <div className="p-6 flex flex-col h-full">
        <div className="flex items-center justify-between mb-8">
          <div className="flex items-center gap-2 text-rose-400">
            <Terminal className="w-5 h-5" />
            <h2 className="text-xl font-heading font-bold">Failure Analysis</h2>
          </div>
          <button onClick={onClose} className="p-2 hover:bg-slate-800 rounded-full transition-colors">
            <X className="w-6 h-6 text-slate-400" />
          </button>
        </div>

        <div className="space-y-6 flex-1 overflow-y-auto">
          <div className="p-4 bg-rose-500/5 border border-rose-500/20 rounded-xl">
            <label className="text-[10px] uppercase tracking-widest text-rose-400 font-bold block mb-1">Reason</label>
            <p className="text-slate-200 font-medium">{failure.reason}</p>
          </div>

          <div className="flex flex-col gap-2">
            <label className="text-[10px] uppercase tracking-widest text-slate-500 font-bold block">Execution Logs</label>
            <pre className="p-4 bg-slate-900 rounded-xl font-mono text-xs text-slate-400 overflow-x-auto border border-slate-800">
              {failure.logs}
            </pre>
          </div>
        </div>

        <button 
          onClick={copyForLLM}
          className="mt-6 w-full py-4 bg-violet-600 hover:bg-violet-500 text-white rounded-xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg shadow-violet-500/20 active:scale-95"
        >
          <Copy className="w-5 h-5" />
          Copy Context for LLM
        </button>
      </div>
    </div>
  );
};
