import React from 'react';
import { CheckCircle2, Clock, AlertCircle, Loader2 } from 'lucide-react';

interface StoryCardProps {
  id: string;
  title: string;
  status: 'pending' | 'running' | 'done' | 'failed';
  phase?: string;
  duration?: number;
  isHero?: boolean;
}

export const StoryCard: React.FC<StoryCardProps> = ({ id, title, status, phase, duration, isHero }) => {
  const getStatusStyles = () => {
    switch (status) {
      case 'running':
        // Champagne gold border, with gold glow
        return 'border-gold-300 bg-gold-400/5 shadow-[0_0_20px_rgba(253,230,138,0.2)]';
      case 'done':
        return 'border-emerald-500/20 bg-emerald-500/5 opacity-80';
      case 'failed':
        return 'border-rose-500 bg-rose-950/20';
      default:
        // Pending: Clear White text, no dimming
        return 'border-galaxy-800 bg-galaxy-900/40 text-white';
    }
  };

  const getStatusIcon = () => {
    switch (status) {
      case 'running': return <Loader2 className="w-5 h-5 text-gold-300 animate-spin" />;
      case 'done': return <CheckCircle2 className="w-5 h-5 text-emerald-500" />;
      case 'failed': return <AlertCircle className="w-5 h-5 text-rose-500" />;
      default: return <Clock className="w-4 h-4 text-slate-500" />;
    }
  };

  return (
    <div className={`rounded-xl border backdrop-blur-md transition-all duration-300 ${getStatusStyles()} ${isHero ? 'p-6' : 'p-4'}`}>
      <div className="flex items-center justify-between gap-4">
        <div className="flex items-center gap-4">
          <span className={`font-mono rounded px-2 py-1 ${isHero ? 'text-sm font-bold bg-galaxy-950/50 text-gold-100' : 'text-xs text-slate-400 bg-galaxy-950/30'}`}>
            {id}
          </span>
          <h3 className={`font-medium ${status === 'pending' ? 'text-white' : 'text-slate-100'} ${isHero ? 'text-xl tracking-tight' : 'text-sm'}`}>
            {title}
          </h3>
        </div>
        <div className="flex items-center gap-3">
          {phase && <span className="text-[10px] uppercase tracking-wider text-slate-400 border border-galaxy-700 px-2 py-0.5 rounded-full bg-galaxy-950/30">{phase}</span>}
          {duration && <span className="text-xs text-slate-500 font-mono">{duration}m</span>}
          {getStatusIcon()}
        </div>
      </div>
    </div>
  );
};
