import React from 'react';
import { Terminal, Cpu, HardDrive, Clock, FolderOpen, ChevronRight, Sparkles, Loader2 } from 'lucide-react';
import { Terminal as TerminalType } from '../types';

interface TerminalCardProps {
  terminal: TerminalType;
  isSelected?: boolean;
  isHero?: boolean;
  onClick?: () => void;
}

const getTypeIcon = (type: string) => {
  switch (type) {
    case 'claude':
      return <Sparkles className="w-4 h-4 text-gold-300" />;
    case 'gemini':
      return <Sparkles className="w-4 h-4 text-blue-400" />;
    case 'codex':
      return <Sparkles className="w-4 h-4 text-emerald-400" />;
    case 'powershell':
      return <Terminal className="w-4 h-4 text-blue-400" />;
    case 'cmd':
      return <Terminal className="w-4 h-4 text-slate-400" />;
    case 'vscode':
      return <Terminal className="w-4 h-4 text-cyan-400" />;
    default:
      return <Terminal className="w-4 h-4 text-slate-500" />;
  }
};

const getTypeLabel = (type: string) => {
  switch (type) {
    case 'claude':
      return 'Claude Code';
    case 'gemini':
      return 'Gemini CLI';
    case 'codex':
      return 'Codex CLI';
    case 'powershell':
      return 'PowerShell';
    case 'cmd':
      return 'CMD';
    case 'vscode':
      return 'VS Code';
    case 'node':
      return 'Node.js';
    default:
      return type;
  }
};

const getStatusColor = (status: string) => {
  switch (status) {
    case 'working':
      return 'bg-gold-400 animate-pulse';
    case 'busy':
      return 'bg-gold-400';
    case 'active':
      return 'bg-emerald-500';
    default:
      return 'bg-slate-600';
  }
};

const formatUptime = (minutes: number): string => {
  if (minutes < 60) return `${minutes}m`;
  const hours = Math.floor(minutes / 60);
  const mins = minutes % 60;
  if (hours < 24) return `${hours}h ${mins}m`;
  const days = Math.floor(hours / 24);
  return `${days}d ${hours % 24}h`;
};

const extractProjectName = (cwd: string): string => {
  const parts = cwd.split(/[/\\]/);
  // Find the project folder (usually after Users\info\)
  const infoIndex = parts.findIndex(p => p.toLowerCase() === 'info');
  if (infoIndex >= 0 && parts[infoIndex + 1]) {
    return parts[infoIndex + 1];
  }
  return parts[parts.length - 1] || cwd;
};

export const TerminalCard: React.FC<TerminalCardProps> = ({
  terminal,
  isSelected,
  isHero,
  onClick,
}) => {
  const statusColor = getStatusColor(terminal.status);
  const projectName = extractProjectName(terminal.cwd);
  const isWorking = terminal.status === 'working';
  const isAI = ['claude', 'gemini', 'codex'].includes(terminal.type);

  const baseClasses = `
    rounded-xl border backdrop-blur-md transition-all duration-300 cursor-pointer
    ${isWorking ? 'border-gold-400 bg-gold-400/15 shadow-[0_0_30px_rgba(253,230,138,0.3)] animate-pulse' : ''}
    ${isSelected && !isWorking ? 'border-gold-300 bg-gold-400/10 shadow-[0_0_20px_rgba(253,230,138,0.2)]' : ''}
    ${!isSelected && !isWorking ? 'border-galaxy-800 bg-galaxy-900/40 hover:border-galaxy-700 hover:bg-galaxy-900/60' : ''}
    ${isHero ? 'p-6' : 'p-4'}
  `;

  return (
    <div className={baseClasses} onClick={onClick}>
      <div className="flex items-start justify-between gap-4">
        {/* Left side: Type and Project */}
        <div className="flex items-start gap-3 flex-1 min-w-0">
          <div className={`p-2 rounded-lg ${terminal.type === 'claude' ? 'bg-gold-400/10' : 'bg-galaxy-800'}`}>
            {getTypeIcon(terminal.type)}
          </div>
          <div className="flex-1 min-w-0">
            <div className="flex items-center gap-2 mb-1">
              <span className={`font-medium ${isHero ? 'text-lg' : 'text-sm'} ${terminal.type === 'claude' ? 'text-gold-100' : 'text-slate-100'}`}>
                {getTypeLabel(terminal.type)}
              </span>
              <span className={`w-2 h-2 rounded-full ${statusColor}`} />
              {isWorking && (
                <span className="flex items-center gap-1 px-2 py-0.5 bg-gold-400/20 rounded-full">
                  <Loader2 className="w-3 h-3 text-gold-300 animate-spin" />
                  <span className="text-[10px] text-gold-300 font-bold uppercase tracking-wider">Working</span>
                </span>
              )}
              <span className="text-[10px] text-slate-500 font-mono">PID {terminal.pid}</span>
            </div>
            <div className="flex items-center gap-1 text-slate-400 text-xs truncate">
              <FolderOpen className="w-3 h-3 flex-shrink-0" />
              <span className="truncate" title={terminal.cwd}>{projectName}</span>
            </div>
          </div>
        </div>

        {/* Right side: Stats */}
        <div className="flex items-center gap-4 flex-shrink-0">
          <div className="text-right">
            <div className="flex items-center gap-1 text-xs text-slate-400">
              <Cpu className="w-3 h-3" />
              <span className={terminal.cpu_percent > 10 ? 'text-gold-300 font-medium' : ''}>
                {terminal.cpu_percent.toFixed(1)}%
              </span>
            </div>
            <div className="flex items-center gap-1 text-xs text-slate-400">
              <HardDrive className="w-3 h-3" />
              <span>{terminal.memory_mb.toFixed(0)}MB</span>
            </div>
          </div>
          <div className="text-right">
            <div className="flex items-center gap-1 text-xs text-slate-400">
              <Clock className="w-3 h-3" />
              <span>{formatUptime(terminal.uptime_minutes)}</span>
            </div>
          </div>
          <ChevronRight className="w-4 h-4 text-slate-600" />
        </div>
      </div>

      {/* Children (if any and is hero) */}
      {isHero && terminal.children.length > 0 && (
        <div className="mt-4 pt-4 border-t border-galaxy-800">
          <div className="text-[10px] uppercase tracking-wider text-slate-500 mb-2">
            Child Processes ({terminal.children.length})
          </div>
          <div className="space-y-1">
            {terminal.children.slice(0, 5).map(child => (
              <div key={child.pid} className="flex items-center justify-between text-xs text-slate-400 bg-galaxy-950/50 px-2 py-1 rounded">
                <span className="truncate flex-1" title={child.cmdline}>
                  {child.name} <span className="text-slate-600">({child.pid})</span>
                </span>
                <span className="flex-shrink-0 ml-2">{child.memory_mb.toFixed(0)}MB</span>
              </div>
            ))}
            {terminal.children.length > 5 && (
              <div className="text-[10px] text-slate-600 pl-2">
                +{terminal.children.length - 5} more...
              </div>
            )}
          </div>
        </div>
      )}

      {/* Full command line (if hero) */}
      {isHero && (
        <div className="mt-4 pt-4 border-t border-galaxy-800">
          <div className="text-[10px] uppercase tracking-wider text-slate-500 mb-2">Command Line</div>
          <pre className="text-xs text-slate-400 font-mono bg-galaxy-950/50 p-2 rounded overflow-x-auto">
            {terminal.cmdline}
          </pre>
        </div>
      )}
    </div>
  );
};
