import React from 'react';
import { LayoutDashboard, Terminal, Sparkles, Cpu, HardDrive } from 'lucide-react';
import { Terminal as TerminalType, SystemStats } from '../types';

interface TerminalSidebarProps {
  terminals: TerminalType[];
  selectedPid: number | null;
  onSelect: (pid: number) => void;
  systemStats: SystemStats | null;
  stats: {
    totalCount: number;
    claudeCount: number;
    totalCpu: number;
    totalMemory: number;
  };
}

const getTypeIcon = (type: string, className: string = "w-4 h-4") => {
  switch (type) {
    case 'claude':
      return <Sparkles className={`${className} text-gold-300`} />;
    default:
      return <Terminal className={`${className} text-slate-400`} />;
  }
};

const extractProjectName = (cwd: string): string => {
  const parts = cwd.split(/[/\\]/);
  const infoIndex = parts.findIndex(p => p.toLowerCase() === 'info');
  if (infoIndex >= 0 && parts[infoIndex + 1]) {
    return parts[infoIndex + 1];
  }
  return parts[parts.length - 1] || cwd;
};

export const TerminalSidebar: React.FC<TerminalSidebarProps> = ({
  terminals,
  selectedPid,
  onSelect,
  systemStats,
  stats,
}) => {
  // Group terminals: Claude first, then others
  const claudeTerminals = terminals.filter(t => t.type === 'claude');
  const otherTerminals = terminals.filter(t => t.type !== 'claude');

  return (
    <div className="w-80 border-r border-gold-200/10 flex flex-col h-full bg-galaxy-950/80 backdrop-blur-xl relative z-20">
      {/* Header */}
      <div className="p-6">
        <div className="flex items-center gap-3 text-gold-200 mb-6 pl-2">
          <LayoutDashboard className="w-6 h-6" />
          <h1 className="text-lg font-heading font-bold tracking-widest text-slate-100 uppercase">
            Mission Control
          </h1>
        </div>

        {/* Quick Stats */}
        <div className="grid grid-cols-2 gap-2 mb-6">
          <div className="bg-galaxy-900/50 border border-galaxy-800 rounded-lg p-3">
            <div className="text-[10px] text-slate-500 uppercase tracking-wide mb-1">Claude</div>
            <div className="text-xl font-bold text-gold-300">{stats.claudeCount}</div>
          </div>
          <div className="bg-galaxy-900/50 border border-galaxy-800 rounded-lg p-3">
            <div className="text-[10px] text-slate-500 uppercase tracking-wide mb-1">Total</div>
            <div className="text-xl font-bold text-slate-300">{stats.totalCount}</div>
          </div>
        </div>
      </div>

      {/* Terminal List */}
      <div className="flex-1 overflow-y-auto px-4 pb-4 scrollbar-none">
        {/* Claude Section */}
        {claudeTerminals.length > 0 && (
          <div className="mb-4">
            <div className="flex items-center gap-2 px-2 mb-2">
              <Sparkles className="w-3 h-3 text-gold-300" />
              <span className="text-[10px] uppercase tracking-wider text-gold-300/80 font-bold">
                Claude Instances
              </span>
            </div>
            <nav className="space-y-1">
              {claudeTerminals.map(t => (
                <button
                  key={t.pid}
                  onClick={() => onSelect(t.pid)}
                  className={`w-full text-left p-3 rounded-xl flex items-center justify-between transition-all group border ${
                    selectedPid === t.pid
                      ? 'bg-gold-glow border-gold-200/40'
                      : 'hover:bg-galaxy-900/50 border-transparent hover:border-galaxy-800'
                  }`}
                >
                  <div className="flex flex-col gap-0.5 min-w-0">
                    <span className={`text-sm font-medium truncate ${selectedPid === t.pid ? 'text-gold-100' : 'text-slate-300 group-hover:text-slate-200'}`}>
                      {extractProjectName(t.cwd)}
                    </span>
                    <span className="text-[10px] text-slate-600 font-mono">
                      PID {t.pid}
                    </span>
                  </div>
                  <div className="flex items-center gap-2">
                    <span className={`w-2 h-2 rounded-full ${t.status === 'busy' ? 'bg-gold-400 animate-pulse' : t.status === 'active' ? 'bg-emerald-500' : 'bg-slate-600'}`} />
                  </div>
                </button>
              ))}
            </nav>
          </div>
        )}

        {/* Other Terminals Section */}
        {otherTerminals.length > 0 && (
          <div>
            <div className="flex items-center gap-2 px-2 mb-2">
              <Terminal className="w-3 h-3 text-slate-500" />
              <span className="text-[10px] uppercase tracking-wider text-slate-500 font-bold">
                Other Terminals
              </span>
            </div>
            <nav className="space-y-1">
              {otherTerminals.slice(0, 10).map(t => (
                <button
                  key={t.pid}
                  onClick={() => onSelect(t.pid)}
                  className={`w-full text-left p-3 rounded-xl flex items-center justify-between transition-all group border ${
                    selectedPid === t.pid
                      ? 'bg-galaxy-800/50 border-galaxy-700'
                      : 'hover:bg-galaxy-900/50 border-transparent hover:border-galaxy-800'
                  }`}
                >
                  <div className="flex flex-col gap-0.5 min-w-0">
                    <span className={`text-sm truncate ${selectedPid === t.pid ? 'text-slate-200' : 'text-slate-400 group-hover:text-slate-300'}`}>
                      {t.type === 'powershell' ? 'PowerShell' : t.type === 'cmd' ? 'CMD' : extractProjectName(t.cwd)}
                    </span>
                    <span className="text-[10px] text-slate-600 font-mono">
                      PID {t.pid}
                    </span>
                  </div>
                  <span className={`w-2 h-2 rounded-full ${t.status === 'busy' ? 'bg-gold-400' : t.status === 'active' ? 'bg-emerald-500' : 'bg-slate-600'}`} />
                </button>
              ))}
              {otherTerminals.length > 10 && (
                <div className="text-[10px] text-slate-600 text-center py-2">
                  +{otherTerminals.length - 10} more terminals
                </div>
              )}
            </nav>
          </div>
        )}
      </div>

      {/* System Stats Footer */}
      <div className="p-4 border-t border-gold-200/5 bg-galaxy-900/20">
        <div className="p-3 bg-galaxy-950/50 rounded-xl border border-galaxy-800">
          <p className="text-[10px] text-slate-500 uppercase font-black mb-2 tracking-wider">System</p>
          {systemStats && (
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2 text-xs text-slate-400">
                  <Cpu className="w-3 h-3" />
                  <span>CPU</span>
                </div>
                <span className={`text-xs font-mono ${systemStats.cpu_percent > 80 ? 'text-rose-400' : systemStats.cpu_percent > 50 ? 'text-gold-300' : 'text-emerald-400'}`}>
                  {systemStats.cpu_percent.toFixed(0)}%
                </span>
              </div>
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2 text-xs text-slate-400">
                  <HardDrive className="w-3 h-3" />
                  <span>Memory</span>
                </div>
                <span className={`text-xs font-mono ${systemStats.memory_percent > 90 ? 'text-rose-400' : systemStats.memory_percent > 70 ? 'text-gold-300' : 'text-emerald-400'}`}>
                  {systemStats.memory_used_gb.toFixed(1)} / {systemStats.memory_total_gb.toFixed(0)} GB
                </span>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};
