import { useState, useEffect, useRef } from 'react';
import { DashboardData, Process, Story, StoryStatus } from '../types';
import mockData from '../data.json';

interface ApiRun {
  process: string;
  run_id: string;
  overall_status: string;
  current_story: string;
  done_story_ids: string[];
  failed_story_ids: string[];
  status_path: string;
  heartbeat_minutes_ago?: number;
}

interface ApiPlan {
  process: string;
  story_ids: string[];
}

export const useDashboardData = () => {
  const [data, setData] = useState<DashboardData>({ processes: [] });
  const [plans, setPlans] = useState<Record<string, string[]>>({});
  const errorRef = useRef<boolean>(false);

  // Helper to fetch plan
  const fetchPlan = async (processId: string) => {
    try {
      const res = await fetch(`/api/process/${processId}/plan`);
      if (res.ok) {
        const plan: ApiPlan = await res.json();
        console.log(`[Dashboard] Fetched plan for ${processId} RAW:`, plan);
        if (plan.story_ids && plan.story_ids.length > 0) {
           setPlans(prev => ({ ...prev, [processId]: plan.story_ids }));
        } else {
           console.warn(`[Dashboard] Plan for ${processId} is empty!`);
        }
      }
    } catch (e) {
      console.error(`Failed to fetch plan for ${processId}`, e);
    }
  };

  useEffect(() => {
    const fetchData = async () => {
      try {
        const res = await fetch('/api/runs');
        if (!res.ok) throw new Error('Failed to fetch runs');
        
        const apiResponse = await res.json();
        const apiRuns: ApiRun[] = apiResponse.runs;

        // Group runs by process
        const processMap = new Map<string, Process>();

        for (const run of apiRuns) {
          // Trigger fetch if plan is missing OR empty
          const existingPlan = plans[run.process];
          if (!existingPlan || existingPlan.length === 0) {
             fetchPlan(run.process);
          }

          const planIds = plans[run.process] || [];
          console.log(`[Hook] Building stories for ${run.process}. Plan length: ${planIds.length}`);
          
          // Construct Stories
          const stories: Story[] = planIds.map(id => {
            let status: StoryStatus = 'pending';
            if (run.done_story_ids.includes(id)) status = 'done';
            if (run.failed_story_ids.includes(id)) status = 'failed';
            if (run.current_story === id) status = 'running';
            
            return {
              id,
              title: `Story ${id}`, 
              status,
            };
          });

          // Fallback: If plan is empty, build from run data
          if (stories.length === 0) {
            run.done_story_ids.forEach(id => stories.push({ id, title: id, status: 'done' }));
            if (run.current_story) stories.push({ id: run.current_story, title: run.current_story, status: 'running' });
            run.failed_story_ids.forEach(id => stories.push({ id, title: id, status: 'failed' }));
          }

          processMap.set(run.process, {
            id: run.process,
            name: run.process.replace(/_/g, ' ').toUpperCase(),
            status: run.overall_status === 'running' ? 'active' : run.overall_status === 'failed' ? 'failed' : 'idle',
            currentRun: {
              id: run.run_id,
              overallStatus: run.overall_status as any,
              lastHeartbeat: new Date().toISOString(),
              stories
            },
            lastFailure: run.overall_status === 'failed' ? {
                runId: run.run_id,
                storyId: run.current_story || 'unknown',
                reason: 'Process reported failure. Check logs.',
                logs: 'Full logs available in terminal or runs folder.'
            } : undefined
          });
        }

        setData({ processes: Array.from(processMap.values()) });
        errorRef.current = false;
      } catch (e) {
        if (!errorRef.current) {
            console.warn("API unavailable, falling back to simulation mode.");
            // Fallback to mock data for design/demo purposes
            setData(mockData as any);
            errorRef.current = true;
        }
      }
    };

    fetchData();
    const interval = setInterval(fetchData, 2000);
    return () => clearInterval(interval);
  }, [plans]); // Re-run when plans update

  return data;
};
