import { useState, useEffect, useCallback, useRef } from 'react';
import { Terminal, SystemStats, TerminalData } from '../types';

interface ApiTerminalsResponse {
  now_epoch: number;
  terminals: Terminal[];
  count: number;
}

const POLL_INTERVAL = 2000; // 2 seconds
const AI_TYPES = ['claude', 'gemini', 'codex'];

// Request notification permission on load
if ('Notification' in window && Notification.permission === 'default') {
  Notification.requestPermission();
}

export const useTerminalData = () => {
  const [terminals, setTerminals] = useState<Terminal[]>([]);
  const [systemStats, setSystemStats] = useState<SystemStats | null>(null);
  const [selectedPid, setSelectedPid] = useState<number | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [lastDoneEvent, setLastDoneEvent] = useState<string | undefined>(undefined);

  // Track previous status of AI terminals to detect "working -> idle" transitions
  const prevStatusRef = useRef<Map<number, string>>(new Map());

  const fetchTerminals = useCallback(async () => {
    try {
      const [terminalsRes, systemRes] = await Promise.all([
        fetch('/api/terminals'),
        fetch('/api/system'),
      ]);

      if (!terminalsRes.ok) {
        throw new Error(`Failed to fetch terminals: ${terminalsRes.status}`);
      }

      const terminalsData: ApiTerminalsResponse = await terminalsRes.json();

      // Filter to show only "interesting" terminals (not child processes of other terminals)
      // Keep: AI terminals (Claude, Gemini, Codex), root powershell/cmd, terminals with children
      const AI_TYPES = ['claude', 'gemini', 'codex'];
      const filtered = terminalsData.terminals.filter(t => {
        // Always show AI terminals (Claude, Gemini, Codex)
        if (AI_TYPES.includes(t.type)) return true;
        // Show PowerShell/CMD that are in Windows Terminal
        if (t.parent_terminal === 'Windows Terminal') return true;
        // Show terminals with children (they are doing work)
        if (t.children.length > 0) return true;
        // Show VS Code terminals
        if (t.type === 'vscode') return true;
        // Skip deep child processes
        return false;
      });

      // Detect "working -> idle/active" transitions for AI terminals
      const prevStatus = prevStatusRef.current;
      for (const t of filtered) {
        if (AI_TYPES.includes(t.type)) {
          const prev = prevStatus.get(t.pid);
          // If was working and now is not working -> done!
          if (prev === 'working' && t.status !== 'working') {
            console.log(`[DONE] ${t.type} (PID ${t.pid}) finished working`);
            setLastDoneEvent(`${t.type}-${t.pid}-${Date.now()}`);

            // Show browser notification
            if ('Notification' in window && Notification.permission === 'granted') {
              new Notification(`${t.type.charAt(0).toUpperCase() + t.type.slice(1)} klaar!`, {
                body: `Terminal in ${t.cwd.split(/[/\\]/).pop()} is klaar met werken.`,
                icon: '/logo.png',
                tag: `terminal-done-${t.pid}`,
              });
            }
          }
          // Update tracked status
          prevStatus.set(t.pid, t.status);
        }
      }

      setTerminals(filtered);
      setError(null);

      if (systemRes.ok) {
        const systemData: SystemStats = await systemRes.json();
        setSystemStats(systemData);
      }

      setIsLoading(false);
    } catch (e) {
      console.error('Error fetching terminal data:', e);
      setError(e instanceof Error ? e.message : 'Unknown error');
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchTerminals();
    const interval = setInterval(fetchTerminals, POLL_INTERVAL);
    return () => clearInterval(interval);
  }, [fetchTerminals]);

  // Group terminals by type for easier rendering
  const claudeTerminals = terminals.filter(t => AI_TYPES.includes(t.type));
  const shellTerminals = terminals.filter(t =>
    t.type === 'powershell' || t.type === 'cmd'
  );
  const otherTerminals = terminals.filter(t =>
    !AI_TYPES.includes(t.type) && t.type !== 'powershell' && t.type !== 'cmd'
  );

  // Get selected terminal details
  const selectedTerminal = selectedPid
    ? terminals.find(t => t.pid === selectedPid)
    : null;

  // Stats
  const totalCpu = terminals.reduce((sum, t) => sum + t.cpu_percent, 0);
  const totalMemory = terminals.reduce((sum, t) => sum + t.memory_mb, 0);

  // Count how many AI terminals are currently working
  const workingCount = claudeTerminals.filter(t => t.status === 'working').length;

  return {
    terminals,
    claudeTerminals,
    shellTerminals,
    otherTerminals,
    systemStats,
    selectedPid,
    setSelectedPid,
    selectedTerminal,
    isLoading,
    error,
    lastDoneEvent,
    workingCount,
    stats: {
      totalCount: terminals.length,
      claudeCount: claudeTerminals.length,
      totalCpu: Math.round(totalCpu * 10) / 10,
      totalMemory: Math.round(totalMemory),
    },
  };
};
